/*--------------------------------------------------------------------*/
/*--- Callgrind                                                    ---*/
/*---                                                       hash.h ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Callgrind, a Valgrind tool for call tracing.

   Copyright (C) 2002-2005, Josef Weidendorfer (Josef.Weidendorfer@gmx.de)

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

/*------------------------------------------------------------------
 * rhash - a generic resizable hash table
 *
 *
 * First, define your own hash entry struct by putting the
 * variables <next> and <hash> at front, like this:
 *
 *  struct my_entry {
 *    // beginning of any rhash entry - do not change
 *    rhash_entry* next;
 *    UWord hash;
 *
 *    // own additions
 *    ...
 *  }
 *
 * Second, provide three callback functions:
 *
 * - a function which calculates a word size hash value from
 *   some data given as void pointer. The hash should cover
 *   the full UWord range, and will be cutted down to hash % size,
 *   where size is the current hash table size.
 *   The function does not need to be provided if the pointer itself
 *   is fine as hash value.
 *
 *     UWord my_hash_value(void* data)
 *     {
 *       return (UWord) ((my_data*)data)->value;
 *     }
 *
 * -  a function to allocate a new hash entry, which is
 *    called with a pointer to some data needed to initialize the
 *    new entry. The same pointer is given to my_hash_value to
 *    initilize the <entry.hash> after calling this function.
 *
 *
 *     rhash_entry* my_new_entry(void* data)
 *     {
 *       my_entry* new = (my_entry*) CLG_MALLOC(sizeof(my_entry));
 *       // Initialize your struct members
 *       ...
 *       return (rhash_entry*) new;
 *     }
 *
 * - a function to check if a given hash entry matches a given key.
 *   If comparing hash values is enough, this function is not needed.
 *
 *  Bool my_has_key(rhash_entry* e, void* data)
 *  {
 *     return e->value == ((my_data*)data)->value;
 *  }
 *
 * Now, declare your hash table, and initialize it:
 *
 *  rhash my_hash = {
 *    .name = "MyHash";            // name (for debugging)
 *    .size = 123;                 // initial size
 *    .hash_value = my_hash_value; // can be 0 if data pointer fits as hash
 *    .has_key = my_has_key;       // can be 0 if hash compare is enough
 *    .new_entry = my_new_entry;
 *  };
 *  ... CLG_(rhash_init)(&my_hash); ...
 *
 * For easy usage of your hash, you can provide wrapper functions
 * to cast types and when you use only a single hash table. Example:
 *
 *  my_entry* get_my_entry(my_data* data)
 *  {
 *     return (my_entry*) CLG_(rhash_get)(&my_hash, (void*) data);
 *  }
 *
 */

#ifndef RHASH_H
#define RHASH_H

#include "global.h"

typedef struct _rhash_entry rhash_entry;
typedef struct _rhash rhash;

struct _rhash_entry {
  rhash_entry* next;
  UWord hash;
};

struct _rhash {
  /* for statistics output */
  Char* name;

  UInt size;
  rhash_entry* (*new_entry)(void*);
  UWord (*hash_value)(void*);
  Bool (*has_key)(rhash_entry*, void*);

  UInt entries;
  UInt resizes;
  rhash_entry** table;
};

/* Initialize a resizable hash table */
void CLG_(rhash_init)(rhash*);

/* Lookup an entry in the hash table
 * Returns the entry or 0 if not found
 */
rhash_entry* CLG_(rhash_lookup)(rhash*, void* data);

/* Lookup an entry in the hash table, and create
 * a new one if not found. It the hash table is 90% full,
 * this automatically doubles the size of the table.
 *
 * The function itself provides no way to detect if a
 * new entry was created. But this can be achieved by
 * initializing an own entry value in my_new_entry
 * and check if this value is set in the returned value.
 *
 * Returns the entry or 0 if allocation failed.
 */
rhash_entry* CLG_(rhash_get)(rhash*, void* data);

/* Remove a hash entry from the table.
 * Returns the removed entry or 0 if not found;
 * you have to free the entry yourself
 */
rhash_entry* CLG_(rhash_remove)(rhash*, void* data);

/* Force resizing of hash table.
 * As removing an entry does not automatically shrink the
 * table, you can do this explicit with this function.
 */
void CLG_(rhash_resize)(rhash*, UInt new_size);

/* Iterate over all hash entries */
void CLG_(rhash_forall)(rhash*, void (*f)(rhash_entry*) );

#endif /* RHASH_H */
