/*
 * Calcurse - text-based organizer
 * Copyright (c) 2004-2005 Frederic Culot
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Send your feedback or comments to : calcurse@culot.org
 * Calcurse home page : http://culot.org/calcurse
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <time.h>

#include "args.h"
#include "vars.h"
#include "event.h"
#include "todo.h"
#include "io.h"

/* 
 * Parse the command-line arguments and call the appropriate
 * routines to handle those arguments. Also initialize the data paths.
 */
int parse_args(int argc, char **argv)
{
	int ch, add_line = 0;
	int unknown_flag = 0, app_found = 0;
	int aflag = 0, cflag = 0, dflag = 0, vflag = 0, hflag = 0, tflag = 0;
	int non_interactive = 0, multiple_flag = 0, load_data = 0;
	char *ddate = "", *cfile = NULL;

	while ((ch = getopt(argc, argv, "hvtad:c:")) != -1) {
		switch (ch) {
		case 'a':
			aflag = 1;
			multiple_flag++;
			load_data++;
			break;
		case 'd':
			dflag = 1;
			multiple_flag++;
			load_data++;
			ddate = optarg;
			break;
		case 'c':
			cflag = 1;
			multiple_flag++;
			load_data++;
			cfile = optarg;
			break;
		case 'h':
			hflag = 1;
			break;
		case 't':
			tflag = 1;
			multiple_flag++;
			load_data++;
			add_line = 1;
			break;
		case 'v':
			vflag = 1;
			break;
		default:
			usage();
			unknown_flag = 1;
			non_interactive = 1;
		}
	}
	argc -= optind;
	argv += optind;
	if (argc >= 1) {	/* incorrect arguments */
		usage();
		return 1;
	} else {
		if (unknown_flag) {
			non_interactive = 1;
		} else if (hflag) {
			help_arg();
			non_interactive = 1;
		} else if (vflag) {
			version_arg();
			non_interactive = 1;
		} else if (multiple_flag) {
			if (load_data) {
				io_init(cfile);
				if (dflag || aflag) load_app();
			}
			if (tflag) {
				todo_arg();
				non_interactive = 1;
			}
			if (dflag) {
				date_arg(ddate, add_line);
				non_interactive = 1;
			} else if (aflag) {
				app_found = app_arg(add_line,0,0,0,0);
				non_interactive = 1;
			}
		} else {
			non_interactive = 0;
			io_init(cfile);
		}
		return non_interactive;
	}
}

/*
 * Print Calcurse version with a short copyright text and exit.
 */
void version_arg()
{
	char vtitle[50];
	char vtext[] =
	    "\nCopyright (c) 2004-2005 Frederic Culot.\nThis is free software; see the source for copying conditions.There is NO\nwarranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n";

	sprintf(vtitle, "Calcurse %s - text-based organizer\n", VERSION);
	fputs(vtitle, stderr);
	fputs(vtext, stderr);
}

/* 
 * Print the command line options and exit.
 */
void help_arg()
{
	char htitle[50];
	sprintf(htitle, "Calcurse %s - text-based organizer\n", VERSION);
	fputs(htitle, stderr);
	fputs(arg_usage, stderr);
	fputs("\nMiscellaneous:\n", stderr);
	fputs("  -h		print this help and exit.\n", stderr);
	fputs("  -v		print calcurse version and exit.\n",
	      stderr);
	fputs("\nOptions:\n", stderr);
	fputs("  -c <file>	specify the calendar <file> to use.\n",
	      stderr);
	fputs("\nNon-interactive:\n", stderr);
	fputs("  -a 		print appointments for current day and exit.\n",
	     stderr);
	fputs("  -d <date|num>	print appointments for <date> or <num> upcoming days and exit.\n",stderr);
	fputs("  -t		print todo list and exit.\n", stderr);
	fputs
	    ("\nFor more information, type '?' from within Calcurse, or read the manpage.\n",
	     stderr);
	fputs
	    ("Mail bug reports and suggestions to <calcurse@culot.org>.\n",
	     stderr);
}

/*
 * Print todo list and exit.
 */
void todo_arg()
{
	struct todo_s *i;
	int nb_tod;

	nb_tod = load_todo();
	fputs("to do:\n",stderr);
	for (i = todolist; i != 0; i = i->next) {
		fputs(" - ",stderr);
		fputs(i->mesg,stderr);
		fputs("\n",stderr);
	}
}

/*
 * Print appointments for given day and exit.
 * If no year, month, and day is given, the given date is used.
 * If there is also no date given, current date is considered.
 */
int app_arg(int add_line, int year, int month, int day, long date)
{
	struct event_s *i;
	struct tm *lt;
	time_t t;
	long today;
	int  print_date = 1, app_found = 0;
	char event_start_time[100];
	char event_end_time[100];
	char date_str[30];

	if (date == 0) {
		today = get_sec_date(year, month, day);
	} else today = date;
	
	/* 
	 * Calculate and print the selected date if there is an app for
 	 * that date and it is the first one, and then print all the apps for
 	 * that date. 
	 */
	for (i = eventlist; i != 0; i = i->next) {
		if (event_inday(i, today)) {
			app_found = 1;
			if (add_line) {
				fputs("\n",stderr);
				add_line = 0;
			}
			if (print_date) {
				t = today;
				lt = localtime(&t);
				sprintf(date_str,"%02u/%02u/%04u",lt->tm_mon+1, 
				    lt->tm_mday, 1900+lt->tm_year);
				fputs(date_str,stderr);
				fputs(":\n",stderr);		
				print_date = 0;
			}
			event_sec2str(i, today, event_start_time,
					event_end_time);
			fputs(" - ",stderr);
			fputs(event_start_time,stderr);
			fputs(" -> ",stderr);
			fputs(event_end_time,stderr); fputs("\n\t",stderr);
			fputs(i->mesg,stderr); fputs("\n",stderr);
		}	
	}
	return app_found;
}

/*
 * Print appointment for the given date or for the given n upcoming
 * days.
 */
void date_arg(char *ddate, int add_line)
{
	int i;
	int year = 0, month = 0, day = 0;
	int numdays = 0, num_digit = 0;
	int arg_len = 0, app_found = 0;
	int date_valid = 0;
	struct tm *ptrtime;
	time_t timer;
	char current_day[3], current_month[3] ,current_year[5];
	long today, ind;
	int sec_in_day = 86400;

	/* 
	 * Check (with the argument length) if a date or a number of days 
	 * was entered, and then call app_arg() to print appointments
	 */
	arg_len = strlen(ddate);
	if (arg_len <= 4) {
		for (i = 0; i <= arg_len-1; i++) { 
			if (isdigit(ddate[i])) num_digit++;	
		}
		if (num_digit == arg_len) numdays = atoi(ddate);

		/* 
		 * Get current date, and print appointments for each day
 		 * in the chosen interval. app_found and add_line are used
 		 * to format the output correctly. 
		 */
		today = get_sec_date(year, month, day);
		ind = today;
		for (i = 0; i < numdays; i++) {
			app_found = app_arg(add_line, 0, 0, 0, ind);
			add_line = app_found;
			ind = ind + sec_in_day;
		}
	} else {
		date_valid = check_date(ddate);
		if (date_valid) {
			sscanf(ddate, "%d / %d / %d", &month, &day, &year);
			app_found = app_arg(add_line, year, month, day, 0);
		} else {
			fputs("Argument to the '-d' flag is not valid\n",stderr);
			fputs("Possible argument formats are : 'mm/dd/yyyy' or 'n'\n",stderr);
			fputs("\nFor more information, type '?' from within Calcurse, or read the manpage.\n",stderr);
			fputs
	    ("Mail bug reports and suggestions to <calcurse@culot.org>.\n",
	     stderr);
		}
	}
}

/*
 * Check if the entered date is of a valid format.
 * First check the format by itself, and then check the 
 * numbers correctness.
 */
int
check_date(char *date)
{
	int ok = 0;
	char month[] = "  ";
	char day[] = "  ";
	char year[] = "    ";

	if ( 
			(strlen(date) == 10) &
			(isdigit(date[0]) != 0) &
			(isdigit(date[1]) != 0) &
			(date[2] == '/') &
			(isdigit(date[3]) != 0) &
			(isdigit(date[4]) != 0) &
			(date[5] == '/') &
			(isdigit(date[6])!=0) & (isdigit(date[7])!=0) & 
			(isdigit(date[8])!=0) & (isdigit(date[9])!=0)
	) {
		strncpy(month, date, 2);
		strncpy(day, date + 3, 2);
		strncpy(year, date + 6, 4);
		if ( (atoi(month) <= 12) & 
		     (atoi(month) >= 1)  &
		     (atoi(day) <= 31) &
		     (atoi(day) >= 1) &
		     (atoi(year) <= 9999) &
		     (atoi(year) > 1))
		ok = 1;		
	}
	return ok;
}

/* 
 * Print Calcurse usage and exit.
 */
void usage()
{
	fputs(arg_usage, stderr);
	fputs("Try 'calcurse -h' for more information.\n", stderr);
}


