/**
* This file is a part of the Cairo-Dock project
*
* Copyright : (C) see the 'copyright' file.
* E-mail    : see the 'copyright' file.
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 3
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <math.h>
#include <string.h>
#include <stdlib.h>

#include "cairo-dock-log.h"
#include "cairo-dock-task.h"

#if (GLIB_MAJOR_VERSION == 2 && GLIB_MINOR_VERSION < 32)
#define CD_GMUTEX pThread->pMutex
#define CD_TASK_GMUTEX pTask->CD_GMUTEX
#else
#define CD_GMUTEX &pThread->aMutex
#define CD_TASK_GMUTEX &pTask->pThread->aMutex
#endif


static gboolean _cairo_dock_timer (CairoDockTask *pTask)
{
	cairo_dock_launch_task (pTask);
	return TRUE;
}

static void cairo_dock_schedule_next_iteration (CairoDockTask *pTask)
{
	if (pTask->iSidTimer == 0)
		pTask->iSidTimer = g_timeout_add_seconds (pTask->iPeriod, (GSourceFunc) _cairo_dock_timer, pTask);
}

static void cairo_dock_cancel_next_iteration (CairoDockTask *pTask)
{
	if (! pTask || pTask->bDiscard) // already did in cairo_dock_discard_task (_free_disc. -> free -> stop -> pause -> _free)
		return;
	if (pTask->iSidTimer != 0)
	{
		g_source_remove (pTask->iSidTimer);
		pTask->iSidTimer = 0;
	}
	if (pTask->get_data != NULL && pTask->pThread) // if there is a thread and no problem
	{
		cd_debug ("Task: Unlock thread if it's running (%p)", pTask);
		// we notify the thread that it can stop
		g_atomic_int_set (&pTask->pThread->iUpdateIsEnded, 0); // just to be sure
		g_atomic_int_set (&pTask->pThread->iThreadCanRun, 0);
		// Maybe the thread is waiting
		if (g_atomic_int_get (&pTask->pThread->iThreadIsRunning) == 0)
		{
			g_mutex_unlock (CD_TASK_GMUTEX); // unlock to stop the thread
			cd_debug ("Task: Unlocked (%p)", pTask);
		}
		pTask->pThread = NULL; // now, if we want to relaunch the task, we'll have to relaunch a new thread
		/// g_mutex_unlock (CD_GMUTEX); // locked at the beginning of the thread or in the thread
		/// g_mutex_trylock (CD_GMUTEX); // if was 
		/// g_mutex_unlock (CD_GMUTEX);
	}
}

static void cairo_dock_perform_task_update (CairoDockTask *pTask)
{
	gboolean bContinue = pTask->update (pTask->pSharedMemory);
	if (! bContinue || pTask->iPeriod == 0) // we can stop the thread if the update function says that we have to stop or if we want a one shot
		cairo_dock_cancel_next_iteration (pTask);
	else
	{
		pTask->iFrequencyState = CAIRO_DOCK_FREQUENCY_NORMAL;
		cairo_dock_schedule_next_iteration (pTask);
	}
}

static void cairo_dock_set_elapsed_time (CairoDockTask *pTask)
{
	pTask->fElapsedTime = g_timer_elapsed (pTask->pClock, NULL);
	g_timer_start (pTask->pClock);
}

static gboolean _cairo_dock_check_for_update (CairoDockTask *pTask)
{
	if (! pTask || pTask->bDiscard || ! pTask->pThread)  // task has been discarded
		return FALSE;
	if (g_atomic_int_get (&pTask->pThread->iThreadIsRunning) == 0)  // data have been produced by the thread
	{
		cd_debug ("Task: Perform task update (%p)", pTask);
		if (pTask->bDiscard)  // task has been discarded
			return FALSE;

		pTask->iSidTimerUpdate = 0; // timer for the update
		// We can perform task update and continue/stop the task's timer.
		cairo_dock_perform_task_update (pTask);

		return FALSE;
	}
	return TRUE; // continue to check if it's possible to perform task update
}

static gpointer _cairo_dock_threaded_calculation (CairoDockTask *pTask)
{
	CairoDockTaskThread *pThread = pTask->pThread; // needed for the thread
	cd_debug ("Task: Start a new thread (%p - %p)", pTask, pThread);
	if (! pThread || ! g_mutex_trylock (CD_GMUTEX)
		|| ! g_atomic_int_compare_and_exchange (&pThread->iThreadIsRunning, 0, 1))
		return NULL; // was locked and is running: should not happen...
	while (g_atomic_int_get (&pThread->iThreadCanRun) == 1)
	{
		if (! pTask || pTask->bDiscard) // discarded just after its launch...
			break;

		cd_debug ("Task: thread: Get data (%p - %p)", pTask, pThread);
		//\_______________________ Get data
		cairo_dock_set_elapsed_time (pTask);
		pTask->get_data (pTask->pSharedMemory);
		g_atomic_int_set (&pThread->iUpdateIsEnded, 1);
		cd_debug ("Task: thread: Data received (%p - %p)", pTask, pThread);
		
		if (g_atomic_int_get (&pThread->iThreadCanRun) == 0 || (! pTask || pTask->bDiscard))
			break; // if the task has been cancelled, we should not wait... we should stop !

		// we launch the update in the main loop
		if (pTask->iSidTimerUpdate == 0)
			pTask->iSidTimerUpdate = g_idle_add ((GSourceFunc) _cairo_dock_check_for_update, pTask); 

		g_atomic_int_set (&pThread->iThreadIsRunning, 0);

		//\_______________________ We lock to wait for the next update
		if (g_atomic_int_get (&pThread->iUpdateIsEnded) == 1) // TODO: is it possible? it takes a few time to launch the update in the main loop + wait for the end + the timer... Maybe... if we stop the task.
			g_mutex_lock (CD_GMUTEX);
	}
	g_atomic_int_set (&pThread->iThreadIsRunning, -1); // the task can be freed

	g_mutex_unlock (CD_GMUTEX); // was locked at the beginning of the thread

	//\______________________ Free
	#if (GLIB_MAJOR_VERSION == 2 && GLIB_MINOR_VERSION < 32)
	g_mutex_free (CD_GMUTEX);
	#else
	g_mutex_clear (CD_GMUTEX);
	#endif
	g_free (pThread);
	cd_debug ("Task: Thread: Stop it (%p - %p)", pTask, pThread);
	return NULL;
}

void cairo_dock_launch_task (CairoDockTask *pTask)
{
	g_return_if_fail (pTask != NULL);
	if (pTask->get_data == NULL)  // no threads, only update
	{
		cairo_dock_set_elapsed_time (pTask);
		cairo_dock_perform_task_update (pTask);
	}
	else
	{
		if (pTask->pThread == NULL) //g_atomic_int_compare_and_exchange (&pTask->iThreadCanRun, 0, 1) // if was 0, now 1 => we can launch a new thread
		{
			GError *error = NULL;

			pTask->pThread = g_new0 (CairoDockTaskThread, 1);
			pTask->pThread->iThreadCanRun = 1;

			#if (GLIB_MAJOR_VERSION == 2 && GLIB_MINOR_VERSION < 32)
			CD_TASK_GMUTEX = g_mutex_new ();
			g_thread_create ((GThreadFunc) _cairo_dock_threaded_calculation, pTask, FALSE, &error);
			#else
			g_mutex_init (CD_TASK_GMUTEX);
			GThread* pThread = g_thread_try_new ("Task", (GThreadFunc) _cairo_dock_threaded_calculation, pTask, &error);
			g_thread_unref (pThread);
			#endif

			if (error != NULL)
			{
				cd_warning (error->message);
				g_error_free (error);
				pTask->pThread->iThreadCanRun = 0;
			}
		}
		else if (pTask->iSidTimerUpdate == 0 // to be sure...
			&& g_atomic_int_compare_and_exchange (&pTask->pThread->iThreadIsRunning, 0, 1)) // we announce that we thread is now running
		{
			g_atomic_int_set (&pTask->pThread->iUpdateIsEnded, 0);
			// the thread is launched but is waiting
			g_mutex_unlock (CD_TASK_GMUTEX); // unlock to get new data
		}
		
		/* if (pTask->iSidTimerUpdate == 0)
			pTask->iSidTimerUpdate = g_timeout_add (MAX (100, MIN (0.10 * pTask->iPeriod, 333)), (GSourceFunc) _cairo_dock_check_for_update, pTask);*/
	}
}


static gboolean _cairo_dock_one_shot_timer (CairoDockTask *pTask)
{
	pTask->iSidTimer = 0;
	cairo_dock_launch_task (pTask);
	return FALSE;
}

void cairo_dock_launch_task_delayed (CairoDockTask *pTask, double fDelay)
{
	cairo_dock_cancel_next_iteration (pTask);
	if (fDelay == 0)
		pTask->iSidTimer = g_idle_add ((GSourceFunc) _cairo_dock_one_shot_timer, pTask);
	else
		pTask->iSidTimer = g_timeout_add (fDelay, (GSourceFunc) _cairo_dock_one_shot_timer, pTask);
}


CairoDockTask *cairo_dock_new_task_full (int iPeriod, CairoDockGetDataAsyncFunc get_data, CairoDockUpdateSyncFunc update, GFreeFunc free_data, gpointer pSharedMemory)
{
	CairoDockTask *pTask = g_new0 (CairoDockTask, 1);
	pTask->iPeriod = iPeriod;
	pTask->get_data = get_data;
	pTask->update = update;
	pTask->free_data = free_data;
	pTask->pSharedMemory = pSharedMemory;
	pTask->pClock = g_timer_new ();
	return pTask;
}


static void _cairo_dock_pause_task (CairoDockTask *pTask)
{
	if (pTask == NULL)
		return ;
	
	cd_debug ("Task: %s (%p)", __func__, pTask);
	cairo_dock_cancel_next_iteration (pTask);
	
	if (pTask->iSidTimerUpdate != 0)
	{
		g_source_remove (pTask->iSidTimerUpdate);
		pTask->iSidTimerUpdate = 0;
	}
}

void cairo_dock_stop_task (CairoDockTask *pTask)
{
	if (pTask == NULL)
		return ;
	
	_cairo_dock_pause_task (pTask);
	
	cd_debug ("Task: Waiting for thread's end..."); ///"(%d)\n", g_atomic_int_get (&pTask->iThreadIsRunning));
	/// while (g_atomic_int_get (&pTask->iThreadIsRunning))
	while (pTask->pThread)
		g_usleep (10);
	cd_debug ("Task: ended.");
}


static gboolean _free_discarded_task (CairoDockTask *pTask)
{
	//g_print ("%s ()\n", __func__);
	cairo_dock_free_task (pTask);
	return FALSE;
}
void cairo_dock_discard_task (CairoDockTask *pTask)
{
	if (pTask == NULL)
		return ;
	
	cd_debug ("Task: %s (%p)", __func__, pTask);
	cairo_dock_cancel_next_iteration (pTask);
	g_atomic_int_set (&pTask->bDiscard, 1);
	
	if (pTask->iSidTimerUpdate == 0)
		pTask->iSidTimerUpdate = g_idle_add ((GSourceFunc) _free_discarded_task, pTask);
}

static void _free_task (CairoDockTask *pTask)
{
	cd_debug ("Task: Free task (%p)", pTask);
	if (pTask->free_data)
		pTask->free_data (pTask->pSharedMemory);
	g_timer_destroy (pTask->pClock);
	g_free (pTask);
}

void cairo_dock_free_task (CairoDockTask *pTask)
{
	if (pTask == NULL)
		return ;
	cairo_dock_stop_task (pTask);
	
	_free_task (pTask);
}

gboolean cairo_dock_task_is_active (CairoDockTask *pTask)
{
	return (pTask != NULL && pTask->iSidTimer != 0);
}

gboolean cairo_dock_task_is_running (CairoDockTask *pTask)
{
	return (pTask != NULL && pTask->iSidTimerUpdate != 0);
}

static void _cairo_dock_restart_timer_with_frequency (CairoDockTask *pTask, int iNewPeriod)
{
	gboolean bNeedsRestart = (pTask->iSidTimer != 0);
	_cairo_dock_pause_task (pTask);
	
	if (bNeedsRestart && iNewPeriod != 0)
		pTask->iSidTimer = g_timeout_add_seconds (iNewPeriod, (GSourceFunc) _cairo_dock_timer, pTask);
}

void cairo_dock_change_task_frequency (CairoDockTask *pTask, int iNewPeriod)
{
	g_return_if_fail (pTask != NULL);
	pTask->iPeriod = iNewPeriod;
	
	_cairo_dock_restart_timer_with_frequency (pTask, iNewPeriod);
}

void cairo_dock_relaunch_task_immediately (CairoDockTask *pTask, int iNewPeriod)
{
	cairo_dock_stop_task (pTask);  // on stoppe avant car on ne veut pas attendre la prochaine iteration.
	if (iNewPeriod >= 0)  // sinon valeur inchangee.
		cairo_dock_change_task_frequency (pTask, iNewPeriod); // nouvelle frequence.
	cairo_dock_launch_task (pTask);  // mesure immediate.
}

void cairo_dock_downgrade_task_frequency (CairoDockTask *pTask)
{
	if (pTask->iFrequencyState < CAIRO_DOCK_FREQUENCY_SLEEP)
	{
		pTask->iFrequencyState ++;
		int iNewPeriod;
		switch (pTask->iFrequencyState)
		{
			case CAIRO_DOCK_FREQUENCY_LOW :
				iNewPeriod = 2 * pTask->iPeriod;
			break ;
			case CAIRO_DOCK_FREQUENCY_VERY_LOW :
				iNewPeriod = 4 * pTask->iPeriod;
			break ;
			case CAIRO_DOCK_FREQUENCY_SLEEP :
				iNewPeriod = 10 * pTask->iPeriod;
			break ;
			default :  // ne doit pas arriver.
				iNewPeriod = pTask->iPeriod;
			break ;
		}
		
		cd_message ("degradation of the frequency (state <- %d/%d)", pTask->iFrequencyState, CAIRO_DOCK_NB_FREQUENCIES-1);
		_cairo_dock_restart_timer_with_frequency (pTask, iNewPeriod);
	}
}

void cairo_dock_set_normal_task_frequency (CairoDockTask *pTask)
{
	if (pTask->iFrequencyState != CAIRO_DOCK_FREQUENCY_NORMAL)
	{
		pTask->iFrequencyState = CAIRO_DOCK_FREQUENCY_NORMAL;
		_cairo_dock_restart_timer_with_frequency (pTask, pTask->iPeriod);
	}
}
