/**
* This file is a part of the Cairo-Dock project
*
* Copyright : (C) see the 'copyright' file.
* E-mail    : see the 'copyright' file.
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 3
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <string.h>
#include <unistd.h>
#define __USE_XOPEN_EXTENDED
#include <stdlib.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>

#include "config.h"
#include "cairo-dock-config.h"
#include "cairo-dock-module-factory.h"
#include "cairo-dock-log.h"
#include "cairo-dock-gui-factory.h"
#include "cairo-dock-icon-facility.h"
#include "cairo-dock-keyfile-utilities.h"
#include "cairo-dock-animations.h"
#include "cairo-dock-dialog-manager.h"
#include "cairo-dock-dock-manager.h"
#include "cairo-dock-dock-factory.h"
#include "cairo-dock-launcher-factory.h"
#include "cairo-dock-desktop-file-factory.h"
#include "cairo-dock-file-manager.h"  // cairo_dock_copy_file
#include "cairo-dock-themes-manager.h"
#include "cairo-dock-applications-manager.h"
#include "cairo-dock-backends-manager.h"
#include "cairo-dock-keybinder.h"
#include "cairo-dock-X-manager.h"
#include "cairo-dock-gui-manager.h"
#include "cairo-dock-gui-factory.h"
#include "cairo-dock-gui-backend.h"
#include "cairo-dock-gui-commons.h"
#include "cairo-dock-gui-items.h"  // cairo_dock_gui_items_get_widget_from_name et cairo_dock_gui_items_update_desklet_params
#include "cairo-dock-gui-simple.h"

#define CAIRO_DOCK_PREVIEW_WIDTH 200
#define CAIRO_DOCK_PREVIEW_HEIGHT 250
#define CAIRO_DOCK_SIMPLE_PANEL_WIDTH 1024
#define CAIRO_DOCK_SIMPLE_PANEL_HEIGHT 700
#define CAIRO_DOCK_SIMPLE_CONF_FILE "cairo-dock-simple.conf"
#define CAIRO_DOCK_SHORTKEY_PAGE 2
#define CAIRO_DOCK_ADDONS_PAGE 3
#define CAIRO_DOCK_THEMES_PAGE 4

static GtkWidget *s_pSimpleConfigWindow = NULL;
static const  gchar *s_cCurrentModuleName = NULL;
static int s_iIconSize;
static int s_iTaskbarType;
static gchar *s_cHoverAnim = NULL;
static gchar *s_cHoverEffect = NULL;
static gchar *s_cClickAnim = NULL;
static gchar *s_cClickEffect = NULL;
static int s_iEffectOnDisappearance = -1;
static gboolean s_bShowThemePage = FALSE;

extern gchar *g_cConfFile;
extern gchar *g_cCurrentThemePath;
extern gchar *g_cCairoDockDataDir;
extern gboolean g_bUseOpenGL;
extern CairoDockDesktopGeometry g_desktopGeometry;

#define cd_reload(module_name) do {\
	pManager = gldi_get_manager (module_name);\
	if (pManager != NULL)\
		gldi_reload_manager_from_keyfile (pManager, pKeyFile);\
	} while (0)

static gchar *_get_animation_name (int i)
{
	switch (i)
	{
		case 0:
			return g_strdup ("bounce");
		case 1:
			return g_strdup ("rotate");
		case 2:
			return g_strdup ("blink");
		case 3:
			return g_strdup ("pulse");
		case 4:
			return g_strdup ("wobbly");
		case 5:
			return g_strdup ("wave");
		case 6:
			return g_strdup ("spot");
		default:
		return g_strdup ("");
	}
}
static const gchar *_get_animation_number (const gchar *s)
{
	if (!s)
		return "";
	if (strcmp (s, "bounce") == 0)
		return "0";
	
	if (strcmp (s, "rotate") == 0)
		return "1";
	
	if (strcmp (s, "blink") == 0)
		return "2";
	
	if (strcmp (s, "pulse") == 0)
		return "3";
	
	if (strcmp (s, "wobbly") == 0)
		return "4";
	
	if (strcmp (s, "wave") == 0)
		return "5";
	
	if (strcmp (s, "spot") == 0)
		return "6";
	return "";
}
static gchar *_get_effect_name (int i)
{
	switch (i)
	{
		case 0:
			return g_strdup ("fire");
		case 1:
			return g_strdup ("stars");
		case 2:
			return g_strdup ("rain");
		case 3:
			return g_strdup ("snow");
		case 4:
			return g_strdup ("storm");
		case 5:
			return g_strdup ("firework");
		default:
		return g_strdup ("");
	}
}
static const gchar *_get_effect_number (const gchar *s)
{
	if (!s)
		return "";
	if (strcmp (s, "fire") == 0)
		return "0";
	
	if (strcmp (s, "stars") == 0)
		return "1";
	
	if (strcmp (s, "rain") == 0)
		return "2";
	
	if (strcmp (s, "snow") == 0)
		return "3";
	
	if (strcmp (s, "storm") == 0)
		return "4";
	
	if (strcmp (s, "firework") == 0)
		return "5";
	return "";
}

static gchar * _make_simple_conf_file (void)
{
	//\_____________ On actualise le fichier de conf simple.
	// on cree le fichier au besoin, et on l'ouvre.
	gchar *cConfFilePath = g_strdup_printf ("%s/%s", g_cCurrentThemePath, CAIRO_DOCK_SIMPLE_CONF_FILE);
	///if (! g_file_test (cConfFilePath, G_FILE_TEST_EXISTS))  // since we remove the opengl effects params, we need to overwrite the file each time. TODO: fix this ...
	{
		cairo_dock_copy_file (CAIRO_DOCK_SHARE_DATA_DIR"/"CAIRO_DOCK_SIMPLE_CONF_FILE, cConfFilePath);
	}
	
	GKeyFile* pSimpleKeyFile = cairo_dock_open_key_file (cConfFilePath);
	g_return_val_if_fail (pSimpleKeyFile != NULL, NULL);
	
	if (cairo_dock_conf_file_needs_update (pSimpleKeyFile, CAIRO_DOCK_VERSION))
	{
		cairo_dock_upgrade_conf_file (cConfFilePath, pSimpleKeyFile, CAIRO_DOCK_SHARE_DATA_DIR"/"CAIRO_DOCK_SIMPLE_CONF_FILE);
		
		g_key_file_free (pSimpleKeyFile);
		pSimpleKeyFile = cairo_dock_open_key_file (cConfFilePath);
		g_return_val_if_fail (pSimpleKeyFile != NULL, NULL);
	}
	
	// comportement
	g_key_file_set_integer (pSimpleKeyFile, "Behavior", "screen border", myDocksParam.iScreenBorder);
	
	g_key_file_set_integer (pSimpleKeyFile, "Behavior", "visibility", myDocksParam.iVisibility);
	
	g_key_file_set_integer (pSimpleKeyFile, "Behavior", "show_on_click", (myDocksParam.bShowSubDockOnClick ? 1 : 0));
	
	g_key_file_set_string (pSimpleKeyFile, "Behavior", "hide effect", myDocksParam.cHideEffect);
	
	int iTaskbarType;
	if (! myTaskbarParam.bShowAppli)
		iTaskbarType = 0;
	else if (myTaskbarParam.bHideVisibleApplis)
		iTaskbarType = 1;
	else if (myTaskbarParam.bGroupAppliByClass)
		iTaskbarType = 2;
	else
		iTaskbarType = 3;
	s_iTaskbarType = iTaskbarType;
	g_key_file_set_integer (pSimpleKeyFile, "Behavior", "taskbar", iTaskbarType);

	g_key_file_set_integer (pSimpleKeyFile, "Behavior", "place icons", myTaskbarParam.iIconPlacement);
	
	// animations
	CairoDockModule *pModule;
	CairoDockModuleInstance *pModuleInstance;
	int iAnimOnMouseHover = -1;
	int iAnimOnClick = -1;
	int iEffectOnMouseHover = -1;
	int iEffectOnClick = -1;
	int iEffectOnDisappearance = 4;
	gsize length;
	pModule = cairo_dock_find_module_from_name ("Animated icons");
	if (pModule != NULL && pModule->pInstancesList != NULL)
	{
		pModuleInstance = pModule->pInstancesList->data;
		GKeyFile *pKeyFile = cairo_dock_open_key_file (pModuleInstance->cConfFilePath);
		if (pKeyFile)
		{
			int *pAnimations = g_key_file_get_integer_list (pKeyFile, "Global", "hover effects", &length, NULL);
			if (pAnimations != NULL)
			{
				iAnimOnMouseHover = pAnimations[0];
				g_free (pAnimations);
			}
			pAnimations = g_key_file_get_integer_list (pKeyFile, "Global", "click launchers", &length, NULL);
			if (pAnimations != NULL)
			{
				iAnimOnClick = pAnimations[0];
				g_free (pAnimations);
			}
			g_key_file_free (pKeyFile);
		}
	}
	
	pModule = cairo_dock_find_module_from_name ("icon effects");
	if (pModule != NULL && pModule->pInstancesList != NULL)
	{
		pModuleInstance = pModule->pInstancesList->data;
		GKeyFile *pKeyFile = cairo_dock_open_key_file (pModuleInstance->cConfFilePath);
		if (pKeyFile)
		{
			int *pAnimations = g_key_file_get_integer_list (pKeyFile, "Global", "effects", &length, NULL);
			if (pAnimations != NULL)
			{
				iEffectOnMouseHover = pAnimations[0];
				g_free (pAnimations);
			}
			pAnimations = g_key_file_get_integer_list (pKeyFile, "Global", "click launchers", &length, NULL);
			if (pAnimations != NULL)
			{
				iEffectOnClick = pAnimations[0];
				g_free (pAnimations);
			}
			g_key_file_free (pKeyFile);
		}
	}
	
	s_iEffectOnDisappearance = -1;
	pModule = cairo_dock_find_module_from_name ("illusion");
	if (pModule != NULL && pModule->pInstancesList != NULL)
	{
		pModuleInstance = pModule->pInstancesList->data;
		GKeyFile *pKeyFile = cairo_dock_open_key_file (pModuleInstance->cConfFilePath);
		if (pKeyFile)
		{
			s_iEffectOnDisappearance = g_key_file_get_integer (pKeyFile, "Global", "disappearance", NULL);
			g_key_file_free (pKeyFile);
		}
	}
	
	g_free (s_cHoverAnim);
	s_cHoverAnim = _get_animation_name (iAnimOnMouseHover);
	g_free (s_cHoverEffect);
	s_cHoverEffect = _get_effect_name (iEffectOnMouseHover);
	g_free (s_cClickAnim);
	s_cClickAnim = _get_animation_name (iAnimOnClick);
	g_free (s_cClickEffect);
	s_cClickEffect = _get_effect_name (iEffectOnClick);
	const gchar *cOnMouseHover[2] = {s_cHoverAnim, s_cHoverEffect};
	const gchar *cOnClick[2] = {s_cClickAnim, s_cClickEffect};
	
	if (g_bUseOpenGL)
		g_key_file_set_string_list (pSimpleKeyFile, "Behavior", "anim_hover", cOnMouseHover, 2);
	else
	{
		g_key_file_remove_comment (pSimpleKeyFile, "Behavior", "anim_hover", NULL);
		g_key_file_remove_key (pSimpleKeyFile, "Behavior", "anim_hover", NULL);
	}
	g_key_file_set_string_list (pSimpleKeyFile, "Behavior", "anim_click", cOnClick, 2);
	if (g_bUseOpenGL)
		g_key_file_set_integer (pSimpleKeyFile, "Behavior", "anim_disappear", s_iEffectOnDisappearance);
	else
	{
		g_key_file_remove_comment (pSimpleKeyFile, "Behavior", "anim_disappear", NULL);
		g_key_file_remove_key (pSimpleKeyFile, "Behavior", "anim_disappear", NULL);
	}
	if (g_bUseOpenGL)
	{
		gchar *cComment = g_key_file_get_comment (pSimpleKeyFile, "Behavior", "anim_disappear", NULL);
		if (cComment != NULL)
		{
			gchar *str = cComment;
			cComment = g_strdup_printf ("l[%s;%s;%s;%s;%s;%s]%s", "Evaporate", "Fade out", "Explode", "Break", "Black Hole", "Random", cComment+1);
			g_free (str);
			g_key_file_set_comment (pSimpleKeyFile, "Behavior", "anim_disappear", cComment, NULL);
			g_free (cComment);
		}
	}
	
	// apparence
	g_key_file_set_string (pSimpleKeyFile, "Appearance", "default icon directory", myIconsParam.cIconTheme);
	
	int iIconSize = cairo_dock_convert_icon_size_to_enum (myIconsParam.iIconWidth);
	iIconSize --;  // skip the "default" enum
	
	g_key_file_set_integer (pSimpleKeyFile, "Appearance", "icon size", iIconSize);
	s_iIconSize = iIconSize;
	
	g_key_file_set_string (pSimpleKeyFile, "Appearance", "main dock view", myBackendsParam.cMainDockDefaultRendererName);
	
	g_key_file_set_string (pSimpleKeyFile, "Appearance", "sub-dock view", myBackendsParam.cSubDockDefaultRendererName);
	
	// applets
	gchar *cLink = cairo_dock_get_third_party_applets_link ();
	g_key_file_set_string (pSimpleKeyFile, "Add-ons", "third party", cLink);
	g_free (cLink);
	
	cairo_dock_write_keys_to_file (pSimpleKeyFile, cConfFilePath);
	
	g_key_file_free (pSimpleKeyFile);
	return cConfFilePath;
}


static void _load_theme (gboolean bSuccess, GtkWidget *pWaitingDialog)
{
	cd_debug ("%s ()", __func__);
	if (s_pSimpleConfigWindow == NULL)  // si l'utilisateur a ferme la fenetre entre-temps, on considere qu'il a abandonne.
	{
		cd_debug ("user has given up");
		return ;
	}
	if (bSuccess)
	{
		cd_debug ("loading new current theme...");
		cairo_dock_load_current_theme ();
		
		cairo_dock_set_status_message (s_pSimpleConfigWindow, "");
		
		cd_debug ("now reload window");
		gchar *cConfFilePath = _make_simple_conf_file ();
		g_free (cConfFilePath);
		s_bShowThemePage = TRUE;
		cairo_dock_reload_generic_gui (s_pSimpleConfigWindow);
	}
	else
	{
		cd_debug ("Could not import the theme.");
		cairo_dock_set_status_message (s_pSimpleConfigWindow, _("Could not import the theme."));
	}
	gtk_widget_destroy (pWaitingDialog);
}
static gboolean on_apply_theme_simple (GtkWidget *pThemeNotebook, gpointer data)
{
	//\_______________ On ouvre le fichier de conf.
	gchar *cConfFilePath = g_strdup_printf ("%s/%s", g_cCurrentThemePath, CAIRO_DOCK_SIMPLE_CONF_FILE);
	GKeyFile* pSimpleKeyFile = cairo_dock_open_key_file (cConfFilePath);
	g_return_val_if_fail (pSimpleKeyFile != NULL, TRUE);
	
	//\_______________ On recupere l'onglet courant.
	int iNumPage = gtk_notebook_get_current_page (GTK_NOTEBOOK (pThemeNotebook));
	pThemeNotebook = gtk_notebook_get_nth_page (GTK_NOTEBOOK (pThemeNotebook), iNumPage);
	
	//\_______________ On effectue les actions correspondantes.
	gboolean bReloadWindow = FALSE;
	switch (iNumPage)
	{
		case 0:  // load a theme
			cairo_dock_set_status_message (s_pSimpleConfigWindow, _("Importing theme..."));
			bReloadWindow = cairo_dock_load_theme (pSimpleKeyFile, (GFunc) _load_theme, s_pSimpleConfigWindow);  // bReloadWindow reste a FALSE, on ne rechargera la fenetre que lorsque le theme aura ete importe.
		break;
		
		case 1:  // save current theme
			bReloadWindow = cairo_dock_save_current_theme (pSimpleKeyFile);
			if (bReloadWindow)
				cairo_dock_set_status_message (s_pSimpleConfigWindow, _("Theme has been saved"));
		break;
		
		case 2:  // delete some themes
			bReloadWindow = cairo_dock_delete_user_themes (pSimpleKeyFile);
			if (bReloadWindow)
				cairo_dock_set_status_message (s_pSimpleConfigWindow, _("Themes have been deleted"));
		break;
	}
	
	//\_______________ On nettoie le fichier de conf si on recharge la fenetre.
	if(bReloadWindow)
	{
		g_key_file_remove_group (pSimpleKeyFile, "Save", NULL);
		g_key_file_remove_group (pSimpleKeyFile, "Delete", NULL);
		cairo_dock_write_keys_to_file (pSimpleKeyFile, cConfFilePath);
		s_bShowThemePage = TRUE;
	}
	g_key_file_free (pSimpleKeyFile);
	g_free (cConfFilePath);
	
	return !bReloadWindow;
}

static gboolean on_apply_config_simple (gpointer data)
{
	//\___________________ On recupere l'onglet courant.
	GtkWidget * pMainVBox= gtk_bin_get_child (GTK_BIN (s_pSimpleConfigWindow));
	GList *children = gtk_container_get_children (GTK_CONTAINER (pMainVBox));
	GtkWidget *pGroupWidget = children->data;
	g_list_free (children);
	int iNumPage = gtk_notebook_get_current_page (GTK_NOTEBOOK (pGroupWidget));
	if (iNumPage == CAIRO_DOCK_THEMES_PAGE)
	{
		CairoDockGroupKeyWidget *myWidget = cairo_dock_gui_find_group_key_widget (s_pSimpleConfigWindow, "Themes", "notebook");
		g_return_val_if_fail (myWidget != NULL && myWidget->pSubWidgetList != NULL, TRUE);
		return on_apply_theme_simple (myWidget->pSubWidgetList->data, data);
	}
	else if (iNumPage == CAIRO_DOCK_SHORTKEY_PAGE || iNumPage == CAIRO_DOCK_ADDONS_PAGE)
	{
		return TRUE;
	}
	
	//\_____________ On actualise le fichier de conf principal.
	// on ouvre les 2 fichiers (l'un en lecture, l'autre en ecriture).
	gchar *cConfFilePath = g_strdup_printf ("%s/%s", g_cCurrentThemePath, CAIRO_DOCK_SIMPLE_CONF_FILE);
	GKeyFile* pSimpleKeyFile = cairo_dock_open_key_file (cConfFilePath);
	g_free (cConfFilePath);
	g_return_val_if_fail (pSimpleKeyFile != NULL, TRUE);
	GKeyFile* pKeyFile = cairo_dock_open_key_file (g_cConfFile);
	g_return_val_if_fail (pKeyFile != NULL, TRUE);
	
	// comportement
	CairoDockPositionType iScreenBorder = g_key_file_get_integer (pSimpleKeyFile, "Behavior", "screen border", NULL);
	g_key_file_set_integer (pKeyFile, "Position", "screen border", iScreenBorder);
	
	int iVisibility = g_key_file_get_integer (pSimpleKeyFile, "Behavior", "visibility", NULL);
	g_key_file_set_integer (pKeyFile, "Accessibility", "visibility", iVisibility);
	
	gchar *cHideEffect = g_key_file_get_string (pSimpleKeyFile, "Behavior", "hide effect", NULL);
	g_key_file_set_string (pKeyFile, "Accessibility", "hide effect", cHideEffect);
	g_free (cHideEffect);
	
	int iShowOnClick = (g_key_file_get_integer (pSimpleKeyFile, "Behavior", "show_on_click", NULL) == 1);
	g_key_file_set_integer (pKeyFile, "Accessibility", "show_on_click", iShowOnClick);
	
	int iTaskbarType = g_key_file_get_integer (pSimpleKeyFile, "Behavior", "taskbar", NULL);
	if (iTaskbarType != s_iTaskbarType)
	{
		gboolean bShowAppli = TRUE, bHideVisible, bCurrentDesktopOnly, bMixLauncherAppli, bGroupAppliByClass;
		switch (iTaskbarType)
		{
			case 0:
				bShowAppli = FALSE;
			break;
			case 1:
				bMixLauncherAppli = TRUE;
				bHideVisible = TRUE;
				bGroupAppliByClass = FALSE;
				bCurrentDesktopOnly = FALSE;
			break;
			case 2:
				bMixLauncherAppli = TRUE;
				bGroupAppliByClass = TRUE;
				bHideVisible = FALSE;
				bCurrentDesktopOnly = FALSE;
			break;
			case 3:
			default:
				bCurrentDesktopOnly = TRUE;
				bMixLauncherAppli = FALSE;
				bGroupAppliByClass = FALSE;
				bHideVisible = FALSE;
			break;
		}
		g_key_file_set_boolean (pKeyFile, "TaskBar", "show applications", bShowAppli);
		if (bShowAppli)  // sinon on inutile de modifier les autres parametres.
		{
			g_key_file_set_boolean (pKeyFile, "TaskBar", "hide visible", bHideVisible);
			g_key_file_set_boolean (pKeyFile, "TaskBar", "current desktop only", bCurrentDesktopOnly);
			g_key_file_set_boolean (pKeyFile, "TaskBar", "mix launcher appli", bMixLauncherAppli);
			g_key_file_set_boolean (pKeyFile, "TaskBar", "group by class", bGroupAppliByClass);
		}
		s_iTaskbarType = iTaskbarType;
	}

	int iPlaceIcons = g_key_file_get_integer (pSimpleKeyFile, "Behavior", "place icons", NULL);
	g_key_file_set_integer (pKeyFile, "TaskBar", "place icons", iPlaceIcons);
	
	// animations
	gsize length;
	gchar **cOnMouseHover = g_key_file_get_string_list (pSimpleKeyFile, "Behavior", "anim_hover", &length, NULL);
	gchar **cOnClick = g_key_file_get_string_list (pSimpleKeyFile, "Behavior", "anim_click", &length, NULL);
	int iEffectOnDisappearance = -1;
	if (g_key_file_has_key (pSimpleKeyFile, "Behavior", "anim_disappear", NULL))
		iEffectOnDisappearance = g_key_file_get_integer (pSimpleKeyFile, "Behavior", "anim_disappear", NULL);
	
	CairoDockModule *pModule;
	CairoDockModuleInstance *pModuleInstanceAnim = NULL;
	CairoDockModuleInstance *pModuleInstanceEffect = NULL;
	CairoDockModuleInstance *pModuleInstanceIllusion = NULL;
	
	if (cOnMouseHover && cOnMouseHover[0])
	{
		if (strcmp (cOnMouseHover[0], s_cHoverAnim) != 0)
		{
			pModule = cairo_dock_find_module_from_name ("Animated icons");
			if (pModule != NULL && pModule->pInstancesList != NULL)
			{
				pModuleInstanceAnim = pModule->pInstancesList->data;
				cairo_dock_update_conf_file (pModuleInstanceAnim->cConfFilePath,
					G_TYPE_STRING, "Global", "hover effects", _get_animation_number (cOnMouseHover[0]),
					G_TYPE_INVALID);
			}
			g_free (s_cHoverAnim);
			s_cHoverAnim = g_strdup (cOnMouseHover[0]);
		}
		if (cOnMouseHover[1] && strcmp (cOnMouseHover[1], s_cHoverEffect) != 0)
		{
			pModule = cairo_dock_find_module_from_name ("icon effects");
			if (pModule != NULL && pModule->pInstancesList != NULL)
			{
				pModuleInstanceEffect = pModule->pInstancesList->data;
				cairo_dock_update_conf_file (pModuleInstanceEffect->cConfFilePath,
					G_TYPE_STRING, "Global", "effects", _get_effect_number (cOnMouseHover[1]),
					G_TYPE_INVALID);
			}
			g_free (s_cHoverEffect);
			s_cHoverEffect = g_strdup (cOnMouseHover[1]);
		}
	}
	if (cOnClick && cOnClick[0])
	{
		if (strcmp (cOnClick[0], s_cClickAnim) != 0)
		{
			pModule = cairo_dock_find_module_from_name ("Animated icons");
			if (pModule != NULL && pModule->pInstancesList != NULL)
			{
				pModuleInstanceAnim = pModule->pInstancesList->data;
				const gchar *cAnimation = _get_animation_number (cOnClick[0]);
				cairo_dock_update_conf_file (pModuleInstanceAnim->cConfFilePath,
					G_TYPE_STRING, "Global", "click launchers", cAnimation,
					G_TYPE_STRING, "Global", "click applis", cAnimation,
					G_TYPE_STRING, "Global", "click applets", cAnimation,
					G_TYPE_INVALID);
			}
			g_free (s_cClickAnim);
			s_cClickAnim = g_strdup (cOnClick[0]);
		}
		if (cOnClick[1] && strcmp (cOnClick[1], s_cClickEffect) != 0)
		{
			pModule = cairo_dock_find_module_from_name ("icon effects");
			if (pModule != NULL && pModule->pInstancesList != NULL)
			{
				pModuleInstanceEffect = pModule->pInstancesList->data;
				const gchar *cAnimation = _get_effect_number (cOnClick[1]);
				cairo_dock_update_conf_file (pModuleInstanceEffect->cConfFilePath,
					G_TYPE_STRING, "Global", "click launchers", cAnimation,
					G_TYPE_STRING, "Global", "click applis", cAnimation,
					G_TYPE_STRING, "Global", "click applets", cAnimation,
					G_TYPE_INVALID);
			}
			g_free (s_cClickEffect);
			s_cClickEffect = g_strdup (cOnClick[1]);
		}
	}
	if (iEffectOnDisappearance != s_iEffectOnDisappearance)
	{
		pModule = cairo_dock_find_module_from_name ("illusion");
		if (pModule != NULL && pModule->pInstancesList != NULL)
		{
			pModuleInstanceIllusion = pModule->pInstancesList->data;
			cairo_dock_update_conf_file (pModuleInstanceIllusion->cConfFilePath,
				G_TYPE_INT, "Global", "disappearance", iEffectOnDisappearance,
				G_TYPE_INT, "Global", "appearance", iEffectOnDisappearance,
				G_TYPE_INVALID);
		}
		s_iEffectOnDisappearance = iEffectOnDisappearance;
	}
	
	g_strfreev (cOnMouseHover);
	g_strfreev (cOnClick);
	
	// apparence
	gchar *cIconTheme = g_key_file_get_string (pSimpleKeyFile, "Appearance", "default icon directory", NULL);
	g_key_file_set_string (pKeyFile, "Icons", "default icon directory", cIconTheme);
	g_free (cIconTheme);
	
	int iIconSize = g_key_file_get_integer (pSimpleKeyFile, "Appearance", "icon size", NULL);
	if (iIconSize != s_iIconSize)
	{
		int iLauncherSize, iIconGap;
		double fMaxScale, fReflectSize;
		iLauncherSize = cairo_dock_convert_icon_size_to_pixels (iIconSize+1, &fMaxScale, &fReflectSize, &iIconGap);  // +1 to skip the "default" enum
		
		gint tab[2] = {iLauncherSize, iLauncherSize};
		g_key_file_set_integer_list (pKeyFile, "Icons", "launcher size", tab, 2);
		tab[0] = myIconsParam.iSeparatorWidth;
		g_key_file_set_integer_list (pKeyFile, "Icons", "separator size", tab, 2);
		g_key_file_set_double (pKeyFile, "Icons", "zoom max", fMaxScale);
		g_key_file_set_double (pKeyFile, "Icons", "field depth", fReflectSize);
		g_key_file_set_integer (pKeyFile, "Icons", "icon gap", iIconGap);
		
		s_iIconSize = iIconSize;
	}
	
	gchar *cMainDockDefaultRendererName = g_key_file_get_string (pSimpleKeyFile, "Appearance", "main dock view", NULL);
	g_key_file_set_string (pKeyFile, "Views", "main dock view", cMainDockDefaultRendererName);
	g_free (cMainDockDefaultRendererName);
	
	gchar *cSubDockDefaultRendererName = g_key_file_get_string (pSimpleKeyFile, "Appearance", "sub-dock view", NULL);
	g_key_file_set_string (pKeyFile, "Views", "sub-dock view", cSubDockDefaultRendererName);
	g_free (cSubDockDefaultRendererName);
	
	// on ecrit tout.
	cairo_dock_write_keys_to_file (pKeyFile, g_cConfFile);
	
	//\_____________ On recharge les modules concernes.
	GldiManager *pManager;
	cd_reload ("Docks");
	cd_reload ("Taskbar");
	cd_reload ("Icons");
	cd_reload ("Backends");
	/**CairoDockInternalModule *pInternalModule;
	cd_reload ("Position");
	cd_reload ("Accessibility");
	cd_reload ("TaskBar");
	cd_reload ("Icons");
	cd_reload ("Views");*/
	if (pModuleInstanceAnim != NULL)
	{
		cairo_dock_reload_module_instance (pModuleInstanceAnim, TRUE);
	}
	if (pModuleInstanceEffect != NULL)
	{
		cairo_dock_reload_module_instance (pModuleInstanceEffect, TRUE);
	}
	if (pModuleInstanceIllusion != NULL)
	{
		cairo_dock_reload_module_instance (pModuleInstanceIllusion, TRUE);
	}
	
	return TRUE;
}

static void on_destroy_config_simple (gpointer data)
{
	s_pSimpleConfigWindow = NULL;
}

static void _cairo_dock_add_one_animation_item (const gchar *cName, CairoDockAnimationRecord *pRecord, GtkListStore *pModele)
{
	GtkTreeIter iter;
	memset (&iter, 0, sizeof (GtkTreeIter));
	gtk_list_store_append (GTK_LIST_STORE (pModele), &iter);
	gtk_list_store_set (GTK_LIST_STORE (pModele), &iter,
		CAIRO_DOCK_MODEL_NAME, (pRecord && pRecord->cDisplayedName != NULL && *pRecord->cDisplayedName != '\0' ? pRecord->cDisplayedName : cName),
		CAIRO_DOCK_MODEL_RESULT, cName, -1);
}
static void _add_one_animation_item (const gchar *cName, CairoDockAnimationRecord *pRecord, GtkListStore *pModele)
{
	if (!pRecord->bIsEffect)
		_cairo_dock_add_one_animation_item (cName, pRecord, pModele);
}
static void _add_one_effect_item (const gchar *cName, CairoDockAnimationRecord *pRecord, GtkListStore *pModele)
{
	if (pRecord->bIsEffect)
		_cairo_dock_add_one_animation_item (cName, pRecord, pModele);
}
static void _make_double_anim_widget (GtkWidget *pSimpleConfigWindow, GKeyFile *pKeyFile, const gchar *cGroupName, const gchar *cKeyName, const gchar *cLabel)
{
	CairoDockGroupKeyWidget *myWidget = cairo_dock_gui_find_group_key_widget (pSimpleConfigWindow, cGroupName, cKeyName);
	if (myWidget == NULL)  // peut arriver vu que en mode cairo on n'a pas "anim_hover"
		return;
	
	gsize length = 0;
	gchar **cValues = g_key_file_get_string_list (pKeyFile, cGroupName, cKeyName, &length, NULL);

	GtkWidget *box = _gtk_hbox_new (CAIRO_DOCK_GUI_MARGIN);
	gtk_box_pack_end (GTK_BOX (myWidget->pKeyBox), box, FALSE, FALSE, 0);
	
	GtkWidget *pLabel = gtk_label_new (_("Animation:"));
	gtk_box_pack_start (GTK_BOX (box), pLabel, FALSE, FALSE, 0);
	
	GtkWidget *pOneWidget = cairo_dock_gui_make_combo (FALSE);
	GtkTreeModel *modele = gtk_combo_box_get_model (GTK_COMBO_BOX (pOneWidget));
	_cairo_dock_add_one_animation_item ("", NULL, GTK_LIST_STORE (modele));
	cairo_dock_foreach_animation ((GHFunc) _add_one_animation_item, modele);
	gtk_box_pack_start (GTK_BOX (box), pOneWidget, FALSE, FALSE, 0);
	cairo_dock_gui_select_in_combo (pOneWidget, cValues[0]);
	myWidget->pSubWidgetList = g_slist_append (myWidget->pSubWidgetList, pOneWidget);
	
	if (g_bUseOpenGL)
	{
		#if (GTK_MAJOR_VERSION < 3)
		pLabel = gtk_vseparator_new ();
		#else
		pLabel = gtk_separator_new (GTK_ORIENTATION_VERTICAL);
		#endif
		gtk_widget_set_size_request (pLabel, 20, 1);
		gtk_box_pack_start (GTK_BOX (box), pLabel, FALSE, FALSE, 0);
		
		pLabel = gtk_label_new (_("Effects:"));
		gtk_box_pack_start (GTK_BOX (box), pLabel, FALSE, FALSE, 0);
		
		pOneWidget = cairo_dock_gui_make_combo (FALSE);
		modele = gtk_combo_box_get_model (GTK_COMBO_BOX (pOneWidget));
		_cairo_dock_add_one_animation_item ("", NULL, GTK_LIST_STORE (modele));
		cairo_dock_foreach_animation ((GHFunc) _add_one_effect_item, modele);
		gtk_box_pack_start (GTK_BOX (box), pOneWidget, FALSE, FALSE, 0);
		cairo_dock_gui_select_in_combo (pOneWidget, cValues[1]);
		myWidget->pSubWidgetList = g_slist_append (myWidget->pSubWidgetList, pOneWidget);
	}
	g_strfreev (cValues);
}
static void _make_theme_manager_widget (GtkWidget *pSimpleConfigWindow)
{
	//\_____________ On recupere notre emplacement perso dans la fenetre.
	CairoDockGroupKeyWidget *myWidget = cairo_dock_gui_find_group_key_widget (pSimpleConfigWindow, "Themes", "notebook");
	g_return_if_fail (myWidget != NULL);
	
	//\_____________ On construit le widget a partir du fichier de conf du theme-manager.
	const gchar *cConfFilePath = CAIRO_DOCK_SHARE_DATA_DIR"/themes.conf";
	GSList *pWidgetList = g_object_get_data (G_OBJECT (pSimpleConfigWindow), "widget-list");
	GPtrArray *pDataGarbage = g_object_get_data (G_OBJECT (pSimpleConfigWindow), "garbage");
	GtkWidget *pThemeNotebook = cairo_dock_build_conf_file_widget (cConfFilePath,
		NULL,
		pSimpleConfigWindow,
		&pWidgetList,
		pDataGarbage,
		NULL);  // les widgets seront ajoutes a la liste deja existante. Donc lors de l'ecriture, ils seront ecrit aussi, dans les cles definies dans le fichier de conf (donc de nouveaux groupes seront ajoutés).
	
	// l'onglet du groupe a deja son propre ascenseur.
	int i, n = gtk_notebook_get_n_pages (GTK_NOTEBOOK (pThemeNotebook));
	for (i = 0; i < n; i ++)
	{
		GtkWidget *pScrolledWindow = gtk_notebook_get_nth_page (GTK_NOTEBOOK (pThemeNotebook), i);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (pScrolledWindow), GTK_POLICY_NEVER, GTK_POLICY_NEVER);
	}
	
	g_object_set (pThemeNotebook, "height-request", MIN (CAIRO_DOCK_SIMPLE_PANEL_HEIGHT, g_desktopGeometry.iXScreenHeight[CAIRO_DOCK_HORIZONTAL] - 100), NULL);  // sinon le notebook est tout petit :-/
	gtk_box_pack_start (GTK_BOX (myWidget->pKeyBox),
		pThemeNotebook,
		TRUE,
		TRUE,
		0);
	myWidget->pSubWidgetList = g_slist_append (myWidget->pSubWidgetList, pThemeNotebook);  // on le met dans la liste, non pas pour recuperer sa valeur (d'ailleurs un notebook n'a pas encore de valeur), mais pour pouvoir y acceder facilement plus tard.
	g_object_set_data (G_OBJECT (pSimpleConfigWindow), "widget-list", pWidgetList);  // on remet la liste, car on prepend les elements dedans, donc son pointeur a change.
	
	//\_____________ On charge les widgets perso de ce fichier.
	cairo_dock_make_tree_view_for_delete_themes (pSimpleConfigWindow);
}
static void _make_modules_widget (GtkWidget *pSimpleConfigWindow)
{
	//\_____________ On recupere notre emplacement perso dans la fenetre.
	CairoDockGroupKeyWidget *myWidget = cairo_dock_gui_find_group_key_widget (pSimpleConfigWindow, "Add-ons", "modules");
	g_return_if_fail (myWidget != NULL);
	
	//\_____________ On construit le tree-view.
	GtkWidget *pOneWidget = cairo_dock_build_modules_treeview ();
	
	//\_____________ On l'ajoute a la fenetre.
	GtkWidget *pScrolledWindow = gtk_scrolled_window_new (NULL, NULL);
	g_object_set (pScrolledWindow, "height-request", MIN (2*CAIRO_DOCK_PREVIEW_HEIGHT, g_desktopGeometry.iXScreenHeight[CAIRO_DOCK_HORIZONTAL] - 175), NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (pScrolledWindow), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (pScrolledWindow), pOneWidget);
	myWidget->pSubWidgetList = g_slist_append (myWidget->pSubWidgetList, pOneWidget);  // on le met dans la liste, non pas pour recuperer sa valeur, mais pour pouvoir y acceder facilement plus tard.
	gtk_box_pack_start (GTK_BOX (myWidget->pKeyBox), pScrolledWindow, FALSE, FALSE, 0);
	
	//\______________ On construit le widget de prevue et on le rajoute a la suite.
	gchar *cDefaultMessage = g_strdup_printf ("<b><span font_desc=\"Sans 14\">%s</span></b>", _("Click on an applet in order to have a preview and a description for it."));
	GPtrArray *pDataGarbage = g_object_get_data (G_OBJECT (pSimpleConfigWindow), "garbage");
	GtkWidget *pPreviewBox = cairo_dock_gui_make_preview_box (pSimpleConfigWindow, pOneWidget, FALSE, 1, cDefaultMessage, CAIRO_DOCK_SHARE_DATA_DIR"/images/"CAIRO_DOCK_LOGO, pDataGarbage);  // vertical packaging.
	gtk_box_pack_start (GTK_BOX (myWidget->pKeyBox), pPreviewBox, FALSE, FALSE, 0);
	g_free (cDefaultMessage);
}
static void _make_shortkeys_widget (GtkWidget *pSimpleConfigWindow)
{
	//\_____________ On recupere notre emplacement perso dans la fenetre.
	CairoDockGroupKeyWidget *myWidget = cairo_dock_gui_find_group_key_widget (pSimpleConfigWindow, "Shortkeys", "shortkeys");
	g_return_if_fail (myWidget != NULL);
	
	//\_____________ On construit le tree-view.
	GtkWidget *pOneWidget = cairo_dock_build_shortkeys_widget ();
	
	//\_____________ On l'ajoute a la fenetre.
	GtkWidget *pScrolledWindow = gtk_scrolled_window_new (NULL, NULL);
	g_object_set (pScrolledWindow, "height-request", MIN (2*CAIRO_DOCK_PREVIEW_HEIGHT, g_desktopGeometry.iXScreenHeight[CAIRO_DOCK_HORIZONTAL] - 175), NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (pScrolledWindow), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (pScrolledWindow), pOneWidget);
	myWidget->pSubWidgetList = g_slist_append (myWidget->pSubWidgetList, pOneWidget);  // on le met dans la liste, non pas pour recuperer sa valeur, mais pour pouvoir y acceder facilement plus tard.
	gtk_box_pack_start (GTK_BOX (myWidget->pKeyBox), pScrolledWindow, FALSE, FALSE, 0);
}
static void _make_widgets (GtkWidget *pSimpleConfigWindow, GKeyFile *pKeyFile)
{
	//\_____________ On ajoute le widget des animations au survol.
	_make_double_anim_widget (pSimpleConfigWindow, pKeyFile, "Behavior", "anim_hover", _("On mouse hover:"));
	//\_____________ On ajoute le widget des animations au clic.
	_make_double_anim_widget (pSimpleConfigWindow, pKeyFile, "Behavior", "anim_click", _("On click:"));
	//\_____________ On ajoute le widget du theme-manager.
	_make_theme_manager_widget (pSimpleConfigWindow);
	//\_____________ On ajoute le widget des modules.
	_make_modules_widget (pSimpleConfigWindow);
	//\_____________ On ajoute le widget des shortkeys.
	_make_shortkeys_widget (pSimpleConfigWindow);
	//\_____________ On montre la page des themes si necessaire.
	if (s_bShowThemePage)
	{
		GtkWidget *pMainVBox = gtk_bin_get_child (GTK_BIN (pSimpleConfigWindow));
		GList *children = gtk_container_get_children (GTK_CONTAINER (pMainVBox));
		g_return_if_fail (children != NULL);
		GtkWidget *pNoteBook = children->data;
		g_list_free (children);
		gtk_widget_show_all (pNoteBook);
		gtk_notebook_set_current_page (GTK_NOTEBOOK (pNoteBook), CAIRO_DOCK_THEMES_PAGE);  // due to historical reasons, GtkNotebook refuses to switch to a page unless the child widget is visible. 
		s_bShowThemePage = FALSE;
	}
}


static GtkWidget * show_main_gui (void)
{
	if (s_pSimpleConfigWindow != NULL)
	{
		gtk_window_present (GTK_WINDOW (s_pSimpleConfigWindow));
		return s_pSimpleConfigWindow;
	}
	
	//\_____________ On construit le fichier de conf simple a partir du .conf normal.
	gchar *cConfFilePath = _make_simple_conf_file ();
	
	//\_____________ On construit la fenetre.
	s_pSimpleConfigWindow = cairo_dock_build_generic_gui_full (cConfFilePath,
		NULL,
		_("Cairo-Dock configuration"),
		CAIRO_DOCK_SIMPLE_PANEL_WIDTH, CAIRO_DOCK_SIMPLE_PANEL_HEIGHT,
		(CairoDockApplyConfigFunc) on_apply_config_simple,
		NULL,
		(GFreeFunc) on_destroy_config_simple,
		_make_widgets,
		NULL);
	
	//\_____________ On ajoute un bouton "mode avance".
	GtkWidget *pMainVBox = gtk_bin_get_child (GTK_BIN (s_pSimpleConfigWindow));
	GList *children = gtk_container_get_children (GTK_CONTAINER (pMainVBox));
	g_return_val_if_fail (children != NULL, s_pSimpleConfigWindow);
	GList *w = g_list_last (children);
	GtkWidget *pButtonsHBox = w->data;
	g_list_free (children);
	
	GtkWidget *pSwitchButton = cairo_dock_make_switch_gui_button ();
	gtk_box_pack_start (GTK_BOX (pButtonsHBox),
		pSwitchButton,
		FALSE,
		FALSE,
		0);
	gtk_box_reorder_child (GTK_BOX (pButtonsHBox), pSwitchButton, 1);
	gtk_widget_show_all (pSwitchButton);
	
	//\_____________ Petit message la 1ere fois.
	gchar *cModeFile = g_strdup_printf ("%s/%s", g_cCairoDockDataDir, ".config-mode");
	if (! g_file_test (cModeFile, G_FILE_TEST_EXISTS))
	{
		g_file_set_contents (cModeFile,
			"0",
			-1,
			NULL);
		Icon *pIcon = cairo_dock_get_current_active_icon ();
		if (pIcon == NULL || pIcon->cParentDockName == NULL || cairo_dock_icon_is_being_removed (pIcon))
			pIcon = cairo_dock_get_dialogless_icon ();
		CairoDock *pDock = cairo_dock_search_dock_from_name (pIcon != NULL ? pIcon->cParentDockName : NULL);
		cairo_dock_show_temporary_dialog_with_default_icon (_("This is the simple configuration panel of Cairo-Dock.\n After you get familiar with it, and if you want to customise your theme\n, you can switch to an advanced mode.\n You can switch from a mode to another at any time."), pIcon, CAIRO_CONTAINER (pDock), 15000);
	}
	g_free (cModeFile);
	
	return s_pSimpleConfigWindow;
}


static void show_module_instance_gui (CairoDockModuleInstance *pModuleInstance, int iShowPage)
{
	cairo_dock_show_items_gui (pModuleInstance->pIcon, NULL, pModuleInstance, iShowPage);
}

static void show_module_gui (const gchar *cModuleName)
{
	CairoDockModule *pModule = cairo_dock_find_module_from_name (cModuleName);
	g_return_if_fail (pModule != NULL);
	
	CairoDockModuleInstance *pModuleInstance = (pModule->pInstancesList != NULL ? pModule->pInstancesList->data : NULL);
	if (pModuleInstance == NULL)  // on n'affiche pas la config d'un module non actif, car il faudrait alors gerer une autre fenetre juste pour eux. Donc seul le mode avance peut le faire.
		return;
	
	cairo_dock_show_items_gui (pModuleInstance->pIcon, NULL, pModuleInstance, -1);
}


static void close_gui (void)
{
	if (s_pSimpleConfigWindow == NULL)
		return;
	
	GtkWidget *pMainVBox = gtk_bin_get_child (GTK_BIN (s_pSimpleConfigWindow));
	GList *children = gtk_container_get_children (GTK_CONTAINER (pMainVBox));
	GList *w = g_list_last (children);
	GtkWidget *pButtonsHBox = w->data;
	g_list_free (children);
	
	children = gtk_container_get_children (GTK_CONTAINER (pButtonsHBox));
	w = g_list_last (children);
	GtkWidget *pQuitButton = w->data;
	g_list_free (children);
	
	gboolean bReturn;
	g_signal_emit_by_name (pQuitButton, "clicked", NULL, &bReturn);
}


static gboolean _update_module_checkbox (GtkTreeModel *pModel, GtkTreePath *path, GtkTreeIter *iter, gpointer *data)
{
	gchar *cWantedModuleName = data[0];
	gchar *cModuleName = NULL;
	gtk_tree_model_get (pModel, iter,
		CAIRO_DOCK_MODEL_RESULT, &cModuleName, -1);
	if (cModuleName && strcmp (cModuleName, cWantedModuleName) == 0)
	{
		gtk_list_store_set (GTK_LIST_STORE (pModel), iter, CAIRO_DOCK_MODEL_ACTIVE, data[1], -1);
		g_free (cModuleName);
		return TRUE;
	}
	g_free (cModuleName);
	return FALSE;
}
static void update_module_state (const gchar *cModuleName, gboolean bActive)
{
	if (s_pSimpleConfigWindow == NULL)
		return;
	g_return_if_fail (cModuleName != NULL);
	
	CairoDockGroupKeyWidget *pGroupKeyWidget = cairo_dock_gui_find_group_key_widget (s_pSimpleConfigWindow, "Add-ons", "modules");
	g_return_if_fail (pGroupKeyWidget != NULL && pGroupKeyWidget->pSubWidgetList != NULL);
	GtkWidget *pOneWidget = pGroupKeyWidget->pSubWidgetList->data;
	
	GtkTreeModel *pModel = gtk_tree_view_get_model (GTK_TREE_VIEW (pOneWidget));
	gpointer data[2] = {(gpointer)cModuleName, GINT_TO_POINTER (bActive)};
	gtk_tree_model_foreach (GTK_TREE_MODEL (pModel), (GtkTreeModelForeachFunc) _update_module_checkbox, data);
}


static void update_modules_list (void)
{
	if (s_pSimpleConfigWindow == NULL)
		return;
	
	CairoDockGroupKeyWidget *pGroupKeyWidget = cairo_dock_gui_find_group_key_widget (s_pSimpleConfigWindow, "Add-ons", "modules");
	g_return_if_fail (pGroupKeyWidget != NULL && pGroupKeyWidget->pSubWidgetList != NULL);
	
	GtkWidget *pOneWidget = pGroupKeyWidget->pSubWidgetList->data;
	GtkTreeModel *pModel = gtk_tree_view_get_model (GTK_TREE_VIEW (pOneWidget));
	g_return_if_fail (pModel != NULL);
	gtk_list_store_clear (GTK_LIST_STORE (pModel));
	
	cairo_dock_foreach_module ((GHRFunc) cairo_dock_add_module_to_modele, pModel);
}


static void update_shortkeys (void)
{
	if (s_pSimpleConfigWindow == NULL)
		return;
	
	CairoDockGroupKeyWidget *pGroupKeyWidget = cairo_dock_gui_find_group_key_widget (s_pSimpleConfigWindow, "Shortkeys", "shortkeys");
	g_return_if_fail (pGroupKeyWidget != NULL && pGroupKeyWidget->pSubWidgetList != NULL);
	
	GtkWidget *pOneWidget = pGroupKeyWidget->pSubWidgetList->data;
	GtkTreeModel *pModel = gtk_tree_view_get_model (GTK_TREE_VIEW (pOneWidget));
	g_return_if_fail (pModel != NULL);
	gtk_list_store_clear (GTK_LIST_STORE (pModel));
	cd_keybinder_foreach ((GFunc) cairo_dock_add_shortkey_to_model, pModel);
}


static void set_status_message_on_gui (const gchar *cMessage)
{
	GtkWidget *pStatusBar = NULL;
	if (s_pSimpleConfigWindow != NULL)
	{
		pStatusBar = g_object_get_data (G_OBJECT (s_pSimpleConfigWindow), "status-bar");
	}
	if (pStatusBar == NULL)
	{
		cairo_dock_gui_items_set_status_message_on_gui (cMessage);
		return;
	}
	gtk_statusbar_pop (GTK_STATUSBAR (pStatusBar), 0);  // clear any previous message, underflow is allowed.
	gtk_statusbar_push (GTK_STATUSBAR (pStatusBar), 0, cMessage);
}


void cairo_dock_register_simple_gui_backend (void)
{
	CairoDockMainGuiBackend *pBackend = g_new0 (CairoDockMainGuiBackend, 1);
	
	pBackend->show_main_gui 				= show_main_gui;
	pBackend->show_module_gui 				= show_module_gui;
	//pBackend->show_module_instance_gui 		= show_module_instance_gui;
	pBackend->close_gui 					= close_gui;
	pBackend->update_module_state 			= update_module_state;
	pBackend->update_module_instance_container = cairo_dock_gui_items_update_module_instance_container;
	pBackend->update_desklet_params 		= cairo_dock_gui_items_update_desklet_params;
	pBackend->update_desklet_visibility_params = cairo_dock_update_desklet_visibility_params;
	pBackend->update_modules_list 			= update_modules_list;
	pBackend->update_shortkeys 				= update_shortkeys;
	pBackend->bCanManageThemes 				= TRUE;
	pBackend->cDisplayedName 				= _("Advanced Mode");
	pBackend->cTooltip 						= _("The advanced mode lets you tweak every single parameter of the dock. It is a powerful tool to customise your current theme.");
	
	cairo_dock_register_config_gui_backend (pBackend);
	
	CairoDockGuiBackend *pConfigBackend = g_new0 (CairoDockGuiBackend, 1);
	
	pConfigBackend->set_status_message_on_gui = set_status_message_on_gui;
	pConfigBackend->reload_current_widget 	= cairo_dock_gui_items_reload_current_widget;
	pConfigBackend->show_module_instance_gui 		= show_module_instance_gui;
	pConfigBackend->get_widget_from_name 	= cairo_dock_gui_items_get_widget_from_name;
	
	cairo_dock_register_gui_backend (pConfigBackend);
}
